/**********************************************************
 * 
 * GenericSingleLinkedList:
 *
 *		Klasa kolekcji, ktra implementuje
 *		struktur danych listy jednokierunkowej
 *		przy uyciu szablonw
 * 
 *      
 *
 *********************************************************/
using System;
using System.Collections;
using System.Collections.Generic; // dla szablonw

namespace Chapter_17
{
    // elementy listy s zarzdzane przez wzy
	public class Node<T>
	{
		public T data;
		public Node<T> next;
	}

    public struct NodeStruct<T>
    {
        public T data;
        public Node<T> next;
    }

	/// <summary>
	/// Uywa interfejsw System.Collection.Generic, aby stworzy
	/// struktur danych listy jednokierunkowej
	/// </summary>
	public class GenericSingleLinkedList<T>: IList<T>, IEnumerable
	{
		// odwouje si do pierwszego elementu Node<T> w licie
		Node<T> head = null;

		// zmienna uywana do synchronizacji listy
		private static object syncObj;

		/// <summary>
		/// Tworzy pola statyczne
		/// </summary>
		static GenericSingleLinkedList()
		{
			// inicjalizacja obiektu, przy uyciu ktrego lista bdzie synchronizowana
			syncObj = new object();
		}

		/// <summary>
		/// nieuywane
		/// </summary>
		public GenericSingleLinkedList()
		{
		}

        #region Elementy IList<T>

 		/// <summary>
		/// Zwraca pozycj na licie dla okrelonego obiektu
		/// </summary>
		/// <param name="value">poszukiwany obiekt</param>
		/// <returns>
		/// pozycja o indeksie rozpoczynajcym si od zera, gdy obiekt zosta znaleziony
		/// lub -1, gdy nie znaleziono
		/// </returns>
        public int IndexOf(T item)
        {
            // ledzenie pooenia
            Node<T> curr = head;
            int count = 0;

            // szukaj obiektu, dopki nie zostanie znaleziony
            //	lub nie osignito koca listy
            while (curr != null && !(curr.data.Equals(item)))
            {
                curr = curr.next;
                count++;
            }

            // koniec listy?
            if (curr != null)
            {
                // zwr pooenie
                return count;
            }

            // wskanik, e obiektu nie znaleziono
            return -1;
        }

        /// <summary>
        /// Dodaj element Node<T> w okrelonej pozycji
        /// </summary>
        /// <param name="index">pozycja, w ktrej zostanie dodany element Node<T></param>
        /// <param name="item">obiekt, do ktrego odwouje si element Node<T></param>
        public void Insert(int index, T item)
        {
            // wstawianie do pustej listy
            if (head == null)
            {
                // mona jedynie doda do pierwszego elementu pustej listy
                if (index > 0)
                {
                    throw new ArgumentOutOfRangeException("index", index, "Wybrana pozycja jest wiksza od rozmiaru listy.");
                }

                // stwrz element Node<T>
                //	i zainicjalizuj jego dane
                head = new Node<T>();
                head.data = item;
                head.next = null;
            }
            else
            {
                // zapamitaj pooenie
                int count = 0;
                Node<T> prev = head;
                Node<T> curr = head;

                // kontynuuj, dopki nie znaleziono pozycji
                //	lub napotkano koniec listy
                while (count < index && curr.next != null)
                {
                    prev = curr;
                    curr = curr.next;
                    count++;
                }

                count++;

                // prba wstawienia poza kocem listy
                if (index > count)
                {
                    throw new ArgumentOutOfRangeException("index", index, "Wybrana pozycja jest wiksza od rozmiaru listy.");
                }

                // wstawienie na kocu listy
                if (curr.next == null)
                {
                    curr.next = new Node<T>();
                    curr.next.data = item;
                    curr.next.next = null;
                }
                else	// wstawienie wewntrz listy
                {
                    prev.next = new Node<T>();
                    prev.next.data = item;
                    prev.next.next = curr;
                }
            }
        }

		/// <summary>
		/// Usuwa element Node<T> z okrelonej pozycji
		/// </summary>
		/// <param name="index">Pozycja usuwanego elementu Node<T></param>
        public void RemoveAt(int index)
        {
            // nie ma sensu usuwanie elementu z pustej listy
            if (head == null)
            {
                throw new ArgumentOutOfRangeException("index", index, "Lista jednokierunkowa jest pusta.");
            }

            // zapamitanie pozycji
            int count = 0;

            // usuwanie pierwszego elementu Node<T>?
            if (count == index)
            {
                head = head.next;
            }
            else	// poszukiwanie elementu Node<T> o pozycji index
            {
                // zapamitanie pooenia aktualnego obiektu Node<T>
                //	oraz znajdujcego si przed nim
                Node<T> prev = head;
                Node<T> curr = head;

                // wykonuj przeszukiwanie, dopki nie uzyskamy 
                //	elementu Node<T> o pozycji rwnej index
                while (count < index)
                {
                    // jeli osignlimy koniec listy, mamy problem
                    if (curr.next == null)
                    {
                        throw new ArgumentOutOfRangeException("index", index, "Wybrana pozycja jest wiksza od rozmiaru listy.");
                    }

                    // zmodyfikuj referencje do poprzedniego
                    //	i aktualnego elementu Node<T>
                    prev = curr;
                    curr = curr.next;

                    // zmodyfikuj pozycj
                    count++;
                }

                // znaleziono poszukiwany element Node<T> - usu go przez
                //	skasowanie z listy
                prev.next = curr.next;
            }
        }

        /// <summary>
        /// Odczytuje i ustawia informacje dla okrelonego pooenia
        /// </summary>
        public T this[int index]
        {
            get
            {
                // Nie ma sensu odczytywanie z pustej listy
                if (head == null)
                {
                    throw new ArgumentOutOfRangeException("index", index, "Lista jednokierunkowa jest pusta.");
                }

                // zapamitaj pozycj
                int count = 0;

                // odzyskujemy pierwszy element Node<T>?
                if (count == index)
                {
                    return head.data;
                }
                else	// szukamy elementu Node<T> o pozycji index
                {
                    // zapamitanie aktualnego elementu Node<T>
                    Node<T> curr = head;

                    // poszukujemy, dopki nie znajdziemy
                    //	elementu Node<T> na pozycji index
                    while (count < index)
                    {
                        // jeli osignlimy koniec listy, mamy problem
                        if (curr.next == null)
                        {
                            throw new ArgumentOutOfRangeException("index", index, "Wybrana pozycja jest wiksza od rozmiaru listy.");
                        }

                        // zmodyfikuj odwoanie do Node<T>
                        curr = curr.next;

                        // zmodyfikuj pooenie
                        count++;
                    }

                    // zwr obiekt
                    return curr.data;
                }
            }
            set
            {
                // nie ma sensu modyfikacja pustej listy
                if (head == null)
                {
                    throw new ArgumentOutOfRangeException("index", index, "Lista jednokierunkowa jest pusta");
                }

                // zapamietanie pooenia
                int count = 0;

                // odczytanie pierwszego elementu Node<T>?
                if (count == index)
                {
                    head.data = value;
                }
                else	// szukamy elementu Node<T> w pozycji index
                {
                    // zapamitaj aktualny element Node<T>
                    Node<T> curr = head;

                    // szukamy, dopki nie znajdziemy 
                    //	elementu Node<T> w pozycji index
                    while (count < index)
                    {
                        // jeli osignlismy koniec listy, mamy problem
                        if (curr.next == null)
                        {
                            throw new ArgumentOutOfRangeException("index", index, "Wybrana pozycja jest wiksza od rozmiaru listy.");
                        }

                        // zmodyfikuj referencj do elementu Node<T>
                        curr = curr.next;

                        // zmodyfikuj pooenie
                        count++;
                    }

                    // zwr obiekt
                    curr.data = value;
                }
            }
        }

        #endregion

        #region Elementy ICollection<T>

        /// <summary>
        /// Dodaje obiekt na koniec listy
        /// </summary>
        /// <param name="item">obiekt do dodania</param>
        public void Add(T item)
        {
            // wstawianie do pustej listy
            if (head == null)
            {
                // stwrz Node<T> 
                //	zainicjalizuj dane
                head = new Node<T>();
                head.data = item;
                head.next = null;
            }
            else
            {
                // zapamitanie pooenia
                Node<T> curr = head;

                // szukaj, dopki nie znaleziono pozycji
                //	lub nie osignito koca listy
                while (curr.next != null)
                {
                    curr = curr.next;
                }

                // wstaw na koniec listy
                curr.next = new Node<T>();
                curr.next.data = item;
                curr.next.next = null;
            }
        }

        /// <summary>
        /// Usuwa wszystkie elementy Node<T> z listy
        /// </summary>
        public void Clear()
        {
            // pierwszy wze na nic nie wskazuje
            head = null;
        }

        /// <summary>
        /// Ustalenie, czy okrelony obiekt jest obecny na licie
        /// </summary>
        /// <param name="item">poszukiwany obiekt</param>
        /// <returns>
        /// true  - obiekt jest na licie
        /// false - obiektu nie ma na licie
        /// </returns>
        public bool Contains(T item)
        {
            // zapamitanie pooenia
            Node<T> curr = head;

            // szukaj, dopki nie znaleziono obiektu
            //	lub nie osignito koca listy
            while (curr != null && !(curr.data.Equals(item)))
            {
                curr = curr.next;
            }

            // koniec listy?
            if (curr != null)
            {
                // obiekt jest na licie
                return true;
            }

            // obiektu nie ma na licie
            return false;
        }

        /// <summary>
        /// Kopiuje list do tablicy w okrelonym pooeniu
        /// </summary>
        /// <param name="array">tablica, do ktrej zostanie skopiowana lista</param>
        /// <param name="arrayIndex">indeks tablicy, od ktrego bd wprowadzane elementy listy</param>
        public void CopyTo(T[] array, int arrayIndex)
        {
            // rozpocznij od pocztku
            Node<T> curr = head;

            // kontynuuj, dopki tablica nie bdzie pena
            //	lub zabraknie kopiowanych elementw
            for (int i = arrayIndex; i < array.Length && curr != null; i++, curr = curr.next)
            {
                array.SetValue(curr.data, i);
            }

            // tablica jest za maa
            if (curr != null)
            {
                throw new ArgumentException("Rozmiar tablicy jest zbyt may.", "array");
            }
        }

        /// <summary>
        /// Zwraca liczb elementw na licie
        /// </summary>
        public int Count
        {
            get
            {
                // zapamitaj pozycj
                Node<T> curr = head;
                int count;

                // iteruj a do koca listy
                for (count = 0; curr != null; count++, curr = curr.next) ;

                // zwr liczb elementw na licie
                return count;
            }
        }

        /// <summary>
        /// Informacja, czy jest to lista tylko do odczytu
        /// </summary>
        public bool IsReadOnly
        {
            get
            {
                //  ta lista ma uprawnienia do zapisu i odczytu
                return false;
            }
        }


        /// <summary>
        /// Usuwa okrelony element z listy
        /// </summary>
        /// <param name="item">obiekt do usunicia</param>
        /// <returns>true, jeli element jest skadnikiem listy; w przeciwnym razie false</returns>
        public bool Remove(T item)
        {
            bool itemExists = true;

            // zapamitaj pooenie
            Node<T> prev = head;
            Node<T> curr = head;

            // poszukuj obiektu, dopki nie zostanie znaleziony
            // lub nie osignito koca listy
            while (curr != null && !(curr.data.Equals(item)))
            {
                prev = curr;
                curr = curr.next;
            }

            // sprawd, czy koniec listy
            if (curr != null)
            {
                // usu element Node<T> z listy
                prev.next = curr.next;
            }
            else
            {
                // warto nieodnaleziona
                itemExists = false;
            }

            return itemExists;
        }

        #endregion

        #region Elementy IEnumerable<T>

        /// <summary>
        /// Zwraca enumerator
        /// </summary>
        /// <returns>GenericSingleLinkedListEnumerator</returns>
        public IEnumerator<T> GetEnumerator()
        {
            return (IEnumerator<T>)new GenericSingleLinkedListEnumerator(head);
        }

        #endregion

        /// <summary>
        /// Enumerator dla kolekcji SingleLinkedList
        /// </summary>
        public class GenericSingleLinkedListEnumerator : IEnumerator<T>, IEnumerator
        {
            Node<T> curr;	// pamita aktualne pooenie
            Node<T> head;	// wskazuje na pierwszy obiekt Node<T>

            // zainicjalizuj enumerator pierwszym elementem
            //	kolekcji SingleLinkedList
            public GenericSingleLinkedListEnumerator(Node<T> head)
            {
                // ustaw pierwszy element
                this.head = head;

                // wyzeruj pooenie
                Reset();
            }

            public void Dispose()
            {
                // zaimplementuj zwalnianie pamici, jeli bdzie to konieczne
            }

            #region Implementation of IEnumerator

            // wyzeruj pooenie na pocztek listy
            public void Reset()
            {
                // ustaw zmienn curr na pozycj zaraz przed pooeniem wskazywanym przez head
                curr = new Node<T>();
                curr.next = head;
            }

            // przesu si do nastpnej pozycji na licie
            public bool MoveNext()
            {
                // upewnij si, e nie jestemy na kocu listy
                if (curr.next != null)
                {
                    // przesu si do nastpnej pozycji
                    curr = curr.next;
                    return true;
                }

                // nie mona wykona ruchu - jestemy na kocu listy
                return false;
            }

            // zwr dane z aktualnej pozycji
            public T Current
            {
                get
                {
                    // zwr dane
                    return curr.data;
                }
            }
            #endregion

			#region Elementy IEnumerator

			object IEnumerator.Current
			{
				get { throw new NotImplementedException(); }
			}

			#endregion
		}

		#region IEnumerable Members

		IEnumerator IEnumerable.GetEnumerator()
		{
			throw new NotImplementedException();
		}

		#endregion
	}
}
